﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.IO;
using System.Text;
using System.Threading;
using System.Windows.Forms;

using KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary;

namespace OpenAPI_App_Manager
{
    public partial class MFPDetails : UserControl
    {        
        private OpenAPIWrapper openAPIHandler;
        public OpenAPIWrapper OpenAPIHandler
        {
            get
            {
                return openAPIHandler;
            }
            set
            {
                openAPIHandler = value;
                if (openAPIHandler == null || openAPIHandler.MFP == null || openAPIHandler.MFP.Address == string.Empty || !openAPIHandler.MFP.OpenAPIEnabled)
                {
                    this.Enabled = false;
                    ClearAll();
                    return;
                }
                else
                {
                    this.Enabled = true;                   
                    GatherDetails();
                }
            }
        }

        public MFPDetails()
        {                    
            InitializeComponent();
        }

        private void Busy(bool AmBusy)
        {
            //Enable or Disable Refresh button
            pictureBoxBusy.Visible = AmBusy;
            buttonRefresh.Visible = !AmBusy;
            this.Refresh();
        }
        
        private void GatherDetails()
        {
            DetailsThread();         
        }
              
        public delegate void GatherMFPDetailsDelegateMethod();
        public void DetailsThread()
        {          
               if(this.InvokeRequired)
                   this.Invoke(new GatherMFPDetailsDelegateMethod(this.GatherDetailsInThread));
               else
                   GatherDetailsInThread();         
        }
      
        private void GatherDetailsInThread()
        {            
            //This is a very long and quite "messy" method.  Essentially, two different kinds of queries are made and the results are
            //used to populate some fields in the UI.            
                try
                {
                    Busy(true);
                    ClearAll();
                     
                    //These booleans will be set to false if retrieving the information fails
                    bool bGotDeviceInfo = true;
                    bool bGotDeviceDescription = true;

                    KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.DeviceDescription devMFP = null;
                    
                    #region GetDeviceInfo
                    // The set of code withing this region block reads the Device Information using SDK Library and stored                 
                    // in a String. This value is not used anywere in the application. Use them as reference for your
                    // application
                    string szDetails = string.Empty; 
                    try
                    {
                        //The first kind of query we use is the MainControlManager.GetDeviceInfo, so we intialise a MainControlManager here
                        KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult cr = openAPIHandler.GetDeviceInfo();
                        if (cr == KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Ack)
                        {
                            //AuthSetting is the current setting of the Authentication mode on the MFP.  For more details on the authentication modes
                            //please see the MFP user documentation
                            switch (openAPIHandler.SetupMainControlManager.DeviceInfo.AuthenticationInformation.AuthSetting)
                            {
                                case "1":
                                    szDetails += "Authentication Setting: None" + Environment.NewLine;
                                    textBoxAuthSetting.Text = "None";
                                    break;
                                case "2":
                                    szDetails += "Authentication Setting: External Server" + Environment.NewLine;
                                    textBoxAuthSetting.Text = "External Server";
                                    break;
                                case "3":
                                    szDetails += "Authentication Setting: MFP Internal User Authentication" + Environment.NewLine;
                                    textBoxAuthSetting.Text = "MFP Internal";
                                    break;
                                case "4":
                                    szDetails += "Authentication Setting: OpenAPI Authentication Application" + Environment.NewLine;
                                    textBoxAuthSetting.Text = "OpenAPI";
                                    break;
                                default:
                                    szDetails += "Authentication Setting: Unknown (" + openAPIHandler.SetupMainControlManager.DeviceInfo.AuthenticationInformation.AuthSetting + ")" + Environment.NewLine;
                                    textBoxAuthSetting.Text = "Unknown (" + openAPIHandler.SetupMainControlManager.DeviceInfo.AuthenticationInformation.AuthSetting + ")";
                                    break;
                            }

                            //Capability.Color determines whether the MFP is capable of outputting colour.
                            //Note that as of June 2008, all current devices do not return a value here, so it is null.
                            try
                            {
                                szDetails += "Colour Capability: " + openAPIHandler.SetupMainControlManager.DeviceInfo.Capability.Color + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "Colour Capability: <<Unknown>>" + Environment.NewLine;
                            }

                            //DeviceInformation.DeviceID is, on "real" MFPs, the MAC address of the MFP's internal network connection, guaranteed to be unique.
                            //With simulators, different results may be returned
                            try
                            {
                                szDetails += "Device ID: " + openAPIHandler.SetupMainControlManager.DeviceInfo.DeviceInformation.DeviceID + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "Device ID: <<Unknown>>" + Environment.NewLine;
                            }

                            //DeviceInformation.ProductID is an identifier for the model of MFP being queried.  It will be the same for each individual
                            //model and differ between models.
                            try
                            {
                                szDetails += "Product ID: " + openAPIHandler.SetupMainControlManager.DeviceInfo.DeviceInformation.ProductID + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "Product ID: <<Unknown>>" + Environment.NewLine;
                            }

                            //DeviceInformation.ProductName is the FULL product name of the MFP being queried, for example: "KONICA MINOLTA bizhub C451"
                            try
                            {
                                szDetails += "Product Name: " + openAPIHandler.SetupMainControlManager.DeviceInfo.DeviceInformation.ProductName + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "Product Name: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Determines whether the MFP has a Hard Disk Drive installed
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.HDD.Installed == "2")
                                {
                                    szDetails += "HDD Installed: True" + Environment.NewLine;
                                    optionStateHDD.Checked = true;
                                }
                                else
                                {
                                    szDetails += "HDD Installed: False" + Environment.NewLine;
                                    optionStateHDD.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "HDD Installed: <<Unknown>>" + Environment.NewLine;
                            }

                            //Determines the MAC address of the MFP.  Note that this SHOULD for a "real" MFP be identical to the DeviceID above
                            try
                            {
                                szDetails += "MAC Address: " + openAPIHandler.SetupMainControlManager.DeviceInfo.NetworkInformation.MacAddress + Environment.NewLine;
                                textBoxMACAddress.Text = openAPIHandler.SetupMainControlManager.DeviceInfo.NetworkInformation.MacAddress;
                            }
                            catch
                            {
                                szDetails += "MAC Address: <<Unknown>>" + Environment.NewLine;
                            }

                            //This returns the OpenAPI PROTOCOL version, not the API version.  (e.g. 101.0)
                            try
                            {
                                szDetails += "OpenAPI Version: " + openAPIHandler.SetupMainControlManager.DeviceInfo.OpenAPIInformation.ProtocolVersion.Major + "." + openAPIHandler.SetupMainControlManager.DeviceInfo.OpenAPIInformation.ProtocolVersion.Minor + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "OpenAPI Version: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //The PanelInformation structure gives various pieces of information about the physical control panel that this MFP has.
                                //It could be useful in more complex applications for determining different workflows, string lengths and so on.
                                //Note that regardless of the panel used, things such as string length are always limited in the same way from a
                                //programmatic perspective - so, a small control panel that can only display a lesser number of characters will still
                                //accept a longer string that would be better suited to a larger control panel.  This is to avoid unnecessary extra
                                //work on behalf of the developer.
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.ColorInformation.Color == "1")
                                {
                                    szDetails += "Colour Control Panel: True" + Environment.NewLine;
                                    szDetails += "Control Panel Number of Colours: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.ColorInformation.Depth + Environment.NewLine;
                                    groupBoxControlPanel.Text = openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.ColorInformation.Depth + " colour Control Panel";
                                }
                                else
                                {
                                    szDetails += "Colour Control Panel: False" + Environment.NewLine;
                                    groupBoxControlPanel.Text = "Black and White Control Panel";
                                }
                                szDetails += "Control Panel Size (logical): " + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.LogicalSize.X + "x" + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.LogicalSize.Y + Environment.NewLine;
                                szDetails += "Control Panel Size (physical): " + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PhysicalSize.X + "x" + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PhysicalSize.Y + Environment.NewLine;
                                string szLogicalX = openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.LogicalSize.X;
                                string szLogicalY = openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.LogicalSize.Y;
                                string szPhysicalX = openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PhysicalSize.X;
                                string szPhysicalY = openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PhysicalSize.Y;
                                //We tidy up the strings here since the X and Y are strings with lots of extra zeroes at the end (e.g. 600.00000)
                                szLogicalX = szLogicalX.Substring(0, szLogicalX.IndexOf("."));
                                szLogicalY = szLogicalY.Substring(0, szLogicalY.IndexOf("."));
                                szPhysicalX = szPhysicalX.Substring(0, szPhysicalX.IndexOf("."));
                                szPhysicalY = szPhysicalY.Substring(0, szPhysicalY.IndexOf("."));
                                labelLogical.Text = szLogicalX + "x" + szLogicalY;
                                labelPhysical.Text = szPhysicalX + "x" + szPhysicalY;
                            }
                            catch
                            {
                                szDetails += "Control Panel Details: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Determines the current language set on the MFP panel.  Note that different firmware releases actually support
                                //different languages, and so one MFP will NOT be capable of all of the languages, however in a "mixed fleet"
                                //potentially any language may be there.  This list can be found in the OpenAPI Class Library documentation
                                string szPanelLang = "";
                                switch (openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PanelLanguageSetting.CurrentLanguage)
                                {
                                    #region Languages
                                    case "1":
                                        szPanelLang = "Japanese";
                                        break;
                                    case "2":
                                        szPanelLang = "English";
                                        break;
                                    case "3":
                                        szPanelLang = "German";
                                        break;
                                    case "4":
                                        szPanelLang = "French";
                                        break;
                                    case "5":
                                        szPanelLang = "Italian";
                                        break;
                                    case "6":
                                        szPanelLang = "Dutch";
                                        break;
                                    case "7":
                                        szPanelLang = "Spanish";
                                        break;
                                    case "8":
                                        szPanelLang = "Danish";
                                        break;
                                    case "9":
                                        szPanelLang = "Czech";
                                        break;
                                    case "10":
                                        szPanelLang = "Portuguese";
                                        break;
                                    case "11":
                                        szPanelLang = "Swedish";
                                        break;
                                    case "12":
                                        szPanelLang = "Norwegian";
                                        break;
                                    case "13":
                                        szPanelLang = "Finnish";
                                        break;
                                    case "14":
                                        szPanelLang = "Hungarian";
                                        break;
                                    case "15":
                                        szPanelLang = "Greek";
                                        break;
                                    case "16":
                                        szPanelLang = "Turkish";
                                        break;
                                    case "17":
                                        szPanelLang = "Cantonese";
                                        break;
                                    case "18":
                                        szPanelLang = "Chinese";
                                        break;
                                    case "19":
                                        szPanelLang = "Korean";
                                        break;
                                    case "20":
                                        szPanelLang = "Russian";
                                        break;
                                    case "21":
                                        szPanelLang = "Croatian";
                                        break;
                                    case "22":
                                        szPanelLang = "Ukrainian";
                                        break;
                                    case "23":
                                        szPanelLang = "Romanian";
                                        break;
                                    case "24":
                                        szPanelLang = "Slovakia";                                        
                                        break;
                                    case "25":
                                        szPanelLang = "Arabic";
                                        break;
                                    case "26":
                                        szPanelLang = "Thai";
                                        break;
                                    case "27":
                                        szPanelLang = "Malay";
                                        break;
                                    case "28":
                                        szPanelLang = "Polish";
                                        break;
                                    case "29":
                                        szPanelLang = "Catalonia";
                                        break;
                                    case "30":
                                        szPanelLang = "Slovene";
                                        break;
                                    default:
                                        szPanelLang = "Other (" + openAPIHandler.SetupMainControlManager.DeviceInfo.PanelInformation.PanelLanguageSetting.CurrentLanguage + ")";
                                        break;
                                    #endregion
                                }
                                szDetails += "Panel Language: " + szPanelLang + Environment.NewLine;
                                textBoxPanelLanguage.Text = szPanelLang;
                            }
                            catch
                            {
                                szDetails += "Panel Language: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //The Interpreters are the print language interpreters supported (e.g. PCL, PostScript etc)
                                //Both the name of the interpreter (LangDescription) and version (LangVersion) are returned
                                //They are returned in an array, so a simple foreach loop can be used to see all of them
                                foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.Common.PrintFunctionInformationNS.InterpreterInfo szInterpreter in openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.InterpreterInfoList.InterpreterInfo)
                                {
                                    szDetails += "Interpreter: " + szInterpreter.LangDescription + " (" + szInterpreter.LangVersion + ")" + Environment.NewLine;
                                    textBoxInterpreters.Text += szInterpreter.LangDescription + " (" + szInterpreter.LangVersion + ")" + Environment.NewLine;
                                }
                            }
                            catch
                            {
                                szDetails += "Interpreters: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Information about the Internet Printing Protocol support on the MFP
                                //For more information on IPP, please consult available material on Wikipedia or elsewhere on the internet
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.Enable == "1")
                                {
                                    szDetails += "IPP: Enabled" + Environment.NewLine;
                                    if (openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.AcceptingJob == "1")
                                        szDetails += "IPP Accept Jobs: True" + Environment.NewLine;
                                    else
                                        szDetails += "IPP Accept Jobs: False" + Environment.NewLine;
                                    szDetails += "IPP Info: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.Info + Environment.NewLine;
                                    szDetails += "IPP Name: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.PrinterName + Environment.NewLine;
                                    szDetails += "IPP Location: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.Location + Environment.NewLine;
                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.Common.PrintFunctionInformationNS.PrinterUri uri in openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.PrinterUriList.PrinterUri)
                                    {
                                        szDetails += "IPP URI: " + uri.Index + ": " + uri.UriInfo + Environment.NewLine;
                                    }
                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.Common.PrintFunctionInformationNS.Operation op in openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.OperationList.Operation)
                                    {
                                        szDetails += "IPP Operation: " + op.Index + ": " + op.Type + " = ";
                                        if (op.Enable == "1")
                                            szDetails += "On";
                                        else
                                            szDetails += "Off";
                                        szDetails += "" + Environment.NewLine;
                                    }
                                    if (openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.IppAuthentication.Enable == "2")
                                    {
                                        szDetails += "IPP Authentication: Enabled" + Environment.NewLine;
                                        szDetails += "IPP Authentication Type: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.IppAuthentication.AuthenticationType + Environment.NewLine;
                                        szDetails += "IPP Authentication UserName: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.IppAuthentication.UserName + Environment.NewLine;
                                        szDetails += "IPP Authentication Password: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.IppAuthentication.Password + Environment.NewLine;
                                        szDetails += "IPP Authentication Realm: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.IppInfo.IppAuthentication.Realm + Environment.NewLine;
                                    }
                                    else
                                    {
                                        szDetails += "IPP Authentication: Disabled" + Environment.NewLine;
                                    }
                                }
                                else
                                {
                                    szDetails += "IPP: Disabled" + Environment.NewLine;
                                }
                            }
                            catch
                            {
                                szDetails += "IPP Info: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Information about the Line Printer Daemon on the MFP
                                //For more information on LPR/LPD, please consult available material on Wikipedia or elsewhere on the internet
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.LprInfo.Enable == "1")
                                {
                                    szDetails += "LPR: Enabled" + Environment.NewLine;
                                    szDetails += "LPR Port Number: " + openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.LprInfo.PortNo + Environment.NewLine;
                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.Common.PrintFunctionInformationNS.Queue qu in openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.LprInfo.QueueList.Queue)
                                    {
                                        szDetails += "LPR Queue: " + qu.Index + ": " + qu.Name + " = ";
                                        if (qu.Enable == "1")
                                            szDetails += "On";
                                        else
                                            szDetails += "Off";
                                        szDetails += "" + Environment.NewLine;
                                    }
                                }
                                else
                                {
                                    szDetails += "LPR: Disabled" + Environment.NewLine;
                                }
                            }
                            catch
                            {
                                szDetails += "LPR Info: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Information about the Raw Printing support on the MFP
                                //For more information on Raw Printing, please consult available material on Wikipedia or elsewhere on the internet
                                foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.Common.PrintFunctionInformationNS.RawPortInfo rawInfo in openAPIHandler.SetupMainControlManager.DeviceInfo.PrintFunctionInformation.PrintProtocol.RawPortInfoList.RawPortInfo)
                                {
                                    szDetails += "Raw Port: " + rawInfo.Index + ": " + rawInfo.PortNo + " = ";
                                    if (rawInfo.Enable == "1")
                                        szDetails += "On";
                                    else
                                        szDetails += "Off";
                                    szDetails += Environment.NewLine;
                                }
                            }
                            catch
                            {
                                szDetails += "Raw Port Info: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //The ScanFunctionInformation structure contains details about the Colour, FileType and Resolution support
                                //on the MFP.  It could be useful, for example, for only showing "valid" options to a user that is creating
                                //limitations on these options.
                                //Note that even if an MFP doesn't support a function, in the Scan API (see other references), you can still
                                //attempt to "control" it and no error will be returned.  e.g. Attempting to "allow" Colour on an MFP that
                                //only allows B/W scanning does not cause an error.  This is so that application developers do not NEED to
                                //take the supported versions in to account with simple applications.  It is however recommended from a user
                                //interface perspective, and this structure can help with that.
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.ColorModeList != null)
                                {
                                    foreach (string cm in openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.ColorModeList.ColorMode)
                                    {
                                        switch (cm)
                                        {
                                            case "1":
                                                szDetails += "Scan Colour Allowed: Full Colour" + Environment.NewLine;
                                                break;
                                            case "2":
                                                szDetails += "Scan Colour Allowed: Black" + Environment.NewLine;
                                                break;
                                            case "3":
                                                szDetails += "Scan Colour Allowed: BiColour" + Environment.NewLine;
                                                break;
                                            case "4":
                                                szDetails += "Scan Colour Allowed: Grayscale" + Environment.NewLine;
                                                break;
                                            case "5":
                                                szDetails += "Scan Colour Allowed: Mono Colour" + Environment.NewLine;
                                                break;
                                            default:
                                                szDetails += "Scan Colour Allowed: Other (" + cm + ")" + Environment.NewLine;
                                                break;
                                        }
                                    }
                                }
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.FileTypeList != null)
                                {
                                    foreach (string ft in openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.FileTypeList.FileType)
                                    {
                                        switch (ft)
                                        {
                                            case "1":
                                                szDetails += "Scan File Type Allowed: PDF" + Environment.NewLine;
                                                break;
                                            case "2":
                                                szDetails += "Scan File Type Allowed: Compact PDF" + Environment.NewLine;
                                                break;
                                            case "3":
                                                szDetails += "Scan File Type Allowed: Jpeg" + Environment.NewLine;
                                                break;
                                            case "4":
                                                szDetails += "Scan File Type Allowed: TIFF" + Environment.NewLine;
                                                break;
                                            case "5":
                                                szDetails += "Scan File Type Allowed: TIFF-S" + Environment.NewLine;
                                                break;
                                            case "6":
                                                szDetails += "Scan File Type Allowed: TIFF-F" + Environment.NewLine;
                                                break;
                                            case "7":
                                                szDetails += "Scan File Type Allowed: TIFF-J" + Environment.NewLine;
                                                break;
                                            default:
                                                szDetails += "Scan File Type Allowed: Other (" + ft + ")" + Environment.NewLine;
                                                break;
                                        }
                                    }
                                }
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.ResolutionList != null)
                                {
                                    foreach (string reso in openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.ResolutionList.Resolution)
                                    {
                                        switch (reso)
                                        {
                                            case "1":
                                                szDetails += "Scan Resolution Allowed: 200x100dpi" + Environment.NewLine;
                                                break;
                                            case "2":
                                                szDetails += "Scan Resolution Allowed: 200x200dpi" + Environment.NewLine;
                                                break;
                                            case "3":
                                                szDetails += "Scan Resolution Allowed: 300x300dpi" + Environment.NewLine;
                                                break;
                                            case "4":
                                                szDetails += "Scan Resolution Allowed: 400x400dpi" + Environment.NewLine;
                                                break;
                                            case "5":
                                                szDetails += "Scan Resolution Allowed: 600x600dpi" + Environment.NewLine;
                                                break;
                                            default:
                                                szDetails += "Scan Resolution Allowed: Other (" + reso + ")" + Environment.NewLine;
                                                break;
                                        }
                                    }
                                }
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.TransferProtocolList != null)
                                {
                                    foreach (string tp in openAPIHandler.SetupMainControlManager.DeviceInfo.ScanFunctionInformation.TransferProtocolList.TransferProtocol)
                                    {
                                        switch (tp)
                                        {
                                            case "1":
                                                szDetails += "Scan Transfer Protocol Allowed: FTP" + Environment.NewLine;
                                                break;
                                            default:
                                                szDetails += "Scan Transfer Protocol Allowed: Other (" + tp + ")" + Environment.NewLine;
                                                break;
                                        }
                                    }
                                }
                            }
                            catch
                            {
                                szDetails += "Scan Function Information: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //The above structure gave information about the scanning abilities of the MFP.  This structure instead gives information
                                //about the PHYSICAL capabilities of the scanner on the MFP.

                                //Whether an ADF (Automatic Document Feeder) is installed
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.Adf == "2") //"2" = true
                                {
                                    szDetails += "ADF Installed: True" + Environment.NewLine;
                                    optionStateADF.Checked = true;
                                }
                                else
                                {
                                    szDetails += "ADF Installed: False" + Environment.NewLine;
                                    optionStateADF.Checked = false;
                                }

                                //Whether the scanner is physically capable of scanning colour
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.Color == "1") //"1" = true
                                    szDetails += "Colour Scanning: True" + Environment.NewLine;
                                else
                                    szDetails += "Colour Scanning: False" + Environment.NewLine;
                                //The physical "reading size" of the scanner
                                szDetails += "Scan Size: " + openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.ReadingSize.Min.X + "x" + openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.ReadingSize.Min.Y + " to " + openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.ReadingSize.Max.X + "x" + openAPIHandler.SetupMainControlManager.DeviceInfo.ScannerInformation.ReadingSize.Max.Y + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "Scanner Information: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //This structure defines the functional capabilities of the MFP

                                //Whether the MFP is capable of being used as a Copier (can it copy?)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.Copy == "2")
                                {
                                    szDetails += "Copy Capable: True" + Environment.NewLine;
                                    optionStateCopyCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "Copy Capable: False" + Environment.NewLine;
                                    optionStateCopyCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "Copy Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as a Printer (can it print?)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.Print == "2")
                                {
                                    szDetails += "Print Capable: True" + Environment.NewLine;
                                    optionStatePrintCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "Print Capable: False" + Environment.NewLine;
                                    optionStatePrintCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "Print Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as a Scanner (can it scan?)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.Scan == "2")
                                {
                                    szDetails += "Scan Capable: True" + Environment.NewLine;
                                    optionStateScanCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "Scan Capable: False" + Environment.NewLine;
                                    optionStateScanCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "Scan Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as a Fax (can it fax?)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.Fax == "2")
                                {
                                    szDetails += "Fax Capable: True" + Environment.NewLine;
                                    optionStateFaxCapable.Checked = true;
                                    if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.Fax2 == "2")
                                    {
                                        szDetails += "Second Line: True" + Environment.NewLine;
                                        optionStateFax2Capable.Checked = true;
                                    }
                                    else
                                    {
                                        szDetails += "Second Line: False" + Environment.NewLine;
                                        optionStateFax2Capable.Checked = false;
                                    }
                                }
                                else
                                {
                                    szDetails += "Fax Capable: False" + Environment.NewLine;
                                    optionStateFaxCapable.Checked = false;
                                    optionStateFax2Capable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "Fax Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as an Internet Fax (see MFP user documentation for details of this function)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.IFax == "2")
                                {
                                    szDetails += "Internet Fax Capable: True" + Environment.NewLine;
                                    optionStateInternetFaxCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "Internet Fax Capable: False" + Environment.NewLine;
                                    optionStateInternetFaxCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "Internet Fax Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as an IP Address Fax (see MFP user documentation for details of this function)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.IpAddressFax == "2")
                                {
                                    szDetails += "IP Address Fax Capable: True" + Environment.NewLine;
                                    optionStateIPAddressFaxCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "IPAddress Fax Capable: False" + Environment.NewLine;
                                    optionStateIPAddressFaxCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "IP Address Fax Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP is capable of being used as a SIP/VoIP Fax (see MFP user documentation for details of this function)
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.SipFax == "2")
                                {
                                    szDetails += "SIP Fax Capable: True" + Environment.NewLine;
                                    optionStateSIPFaxCapable.Checked = true;
                                }
                                else
                                {
                                    szDetails += "SIP Fax Capable: False" + Environment.NewLine;
                                    optionStateSIPFaxCapable.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "SIP Fax Capable: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //Whether the MFP has a "USB Host Board" installed - this board allows other options to be connected
                                if (openAPIHandler.SetupMainControlManager.DeviceInfo.SupportFunction.UsbHostBoard == "2")
                                {
                                    szDetails += "USB Host Board: True" + Environment.NewLine;
                                    optionStateUSBHostBoard.Checked = true;
                                }
                                else
                                {
                                    szDetails += "USB Host Board: False" + Environment.NewLine;
                                    optionStateUSBHostBoard.Checked = false;
                                }
                            }
                            catch
                            {
                                szDetails += "USB Host Board: <<Unknown>>" + Environment.NewLine;
                            }
                            try
                            {
                                //This structure contains details of the User Contact Information that the administrator can enter on the MFP's
                                //webpage.
                                szDetails += "User Contact: " + openAPIHandler.SetupMainControlManager.DeviceInfo.UserContact.Contact + Environment.NewLine;
                                szDetails += "User Contact Name: " + openAPIHandler.SetupMainControlManager.DeviceInfo.UserContact.Name + Environment.NewLine;
                                szDetails += "User Contact Location: " + openAPIHandler.SetupMainControlManager.DeviceInfo.UserContact.Location + Environment.NewLine;
                                szDetails += "User Contact Number: " + openAPIHandler.SetupMainControlManager.DeviceInfo.UserContact.InternalNumber + Environment.NewLine;
                            }
                            catch
                            {
                                szDetails += "User Contact Details: <<Unknown>>" + Environment.NewLine;
                            }
                        }
                        else
                        {
                            bGotDeviceInfo = false;
                            szDetails += "Device Info: <<Unable to get data>>";
                        }
                        //This is the end of all of the details retrievable from the MainControlManager.GetDeviceInfo method.
                        szDetails += Environment.NewLine;
                    }
                    catch(Exception)
                    {
                        bGotDeviceInfo = false;
                    }
                    #endregion
                    #region GetDeviceDescription
                    try
                    {
                        //Our second way of retrieving information is the "DeviceDescriptionManager".
                        //This method is somewhat "simpler" because no MainControlManager is required.  That also means that you do not need to know
                        //anything other than the IP address of the MFP.
                        //The call to the MFP is actually done via HTTP - not OpenAPI.  The DeviceDescription is an XML structure that the MFP can
                        //give.  It is available as the raw XML from http://<address>/OpenAPI/DeviceDescription/
                        //This method simply makes it easier than manually getting it via HTTP and then parsing it.
                        //There is less information returned using this compared to the MainControlManager.GetDeviceInfo method, however some items
                        //are available via this method that are NOT available in GetDeviceInfo.  A common use of DeviceDescriptionManager is to
                        //determine the SSL state and Port number that an MFP is set to before setting up a MainControlManager.  This is used in
                        //the other Sample applications available from Konica Minolta Europe.

                        //First we initialise our DeviceDescriptionManager - note that we also initialise a "DeviceDescriptionSetting" where we specify
                        //the IP address of the MFP.
                        DeviceDescriptionManager DeviceDescriptionMain = new DeviceDescriptionManager();
                        KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionSettingNS.DeviceDescriptionSetting setting = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionSettingNS.DeviceDescriptionSetting();
                        setting.IPAddress = openAPIHandler.MFP.Address;
                        devMFP = new KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.DeviceDescription();

                        KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.ResultStatusNS.ResultStatus ret = DeviceDescriptionMain.GetDeviceDescription(setting, ref devMFP);
                        if (ret.ResultStatusInfo == "1")
                        {
                            //Color.Printer refers to BOTH copying and printing (it is the physical capability of the MFP to do colour)
                            szDetails += "Can Print Colour: " + devMFP.Configuration.DeviceConfiguration.Color.Printer.ToString() + Environment.NewLine;
                            szDetails += "Can Scan Colour: " + devMFP.Configuration.DeviceConfiguration.Color.Scanner.ToString() + Environment.NewLine;
                            optionStateColourPrint.Checked = devMFP.Configuration.DeviceConfiguration.Color.Printer;
                            optionStateColourScan.Checked = devMFP.Configuration.DeviceConfiguration.Color.Scanner;
                            //The "Company Name" defines the company that released the product.  It is usually "KONICA MINOLTA", however may be different
                            //for a product that is OEM'd by another company.
                            szDetails += "Company Name: " + devMFP.Configuration.DeviceConfiguration.CompanyName + Environment.NewLine;
                            //For the following two items, the values are identical to those described above in GetDeviceInfo
                            szDetails += "Product ID: " + devMFP.Configuration.DeviceConfiguration.ProductID + Environment.NewLine;
                            szDetails += "Product Name: " + devMFP.Configuration.DeviceConfiguration.ProductName + Environment.NewLine;
                            textBoxProductName.Text = devMFP.Configuration.DeviceConfiguration.ProductName;
                            //Read SupportFunction if SupportFunctionList is not null 
                            if (devMFP.Configuration.DeviceConfiguration.SupportFunctionList != null && devMFP.Configuration.DeviceConfiguration.SupportFunctionList.SupportFunction !=null)
                            {
                                foreach (KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.SupportFunction func in devMFP.Configuration.DeviceConfiguration.SupportFunctionList.SupportFunction)
                                {
                                    //Supported Functions are "extra" functions that the MFP has.  If there is a specific extra function that you need
                                    //to check for, the documentation for that function should reference this item.
                                    szDetails += "Supported Function: " + func.FunctionName + " = " + func.Enable + Environment.NewLine;
                                }
                            }
                            //If "External OpenAPI" is disabled, the MFP will not answer OpenAPI requests by applications that you develop.
                            szDetails += "External OpenAPI: " + devMFP.Configuration.OpenApiConfiguration.Type + Environment.NewLine;
                            if (devMFP.Configuration.OpenApiConfiguration.Type == "On")
                                optionStateExternalOpenAPIEnabled.Checked = true;
                            textBoxOpenAPIPort.Text = devMFP.Configuration.OpenApiConfiguration.PortNo.ToString();
                            optionStateExternalOpenAPISSL.Checked = devMFP.Configuration.OpenApiConfiguration.Ssl;
                            szDetails += "OpenAPI SSL Enabled: " + devMFP.Configuration.OpenApiConfiguration.Ssl.ToString() + Environment.NewLine;
                            szDetails += "OpenAPI Port Number: " + devMFP.Configuration.OpenApiConfiguration.PortNo.ToString() + Environment.NewLine;
                            //Read SupportVersion if SupportVersionList is not null 
                            if (devMFP.Configuration.SupportVersionList!=null)
                            {
                                foreach (KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.SupportVersion ver in devMFP.Configuration.SupportVersionList.SupportVersion)
                                {
                                    szDetails += "Supported Version: OpenAPI Implementation Version: " + ver.OpenApiVersion.ImplementVersion.Core + " : " + ver.OpenApiVersion.ImplementVersion.DeviceCustomize + Environment.NewLine;
                                    szDetails += "Supported Version: OpenAPI Protocol Version: " + ver.OpenApiVersion.ProtocolVersion.Major.ToString() + "." + ver.OpenApiVersion.ProtocolVersion.Minor.ToString() + Environment.NewLine;
                                    szDetails += "Supported Version: WSDL Implementation Version: " + ver.WsdlVersion.ImplementVersion.Core + " : " + ver.WsdlVersion.ImplementVersion.DeviceCustomize + Environment.NewLine;
                                    szDetails += "Supported Version: WSDL Protocol Version: " + ver.WsdlVersion.ProtocolVersion.Major.ToString() + "." + ver.WsdlVersion.ProtocolVersion.Minor.ToString() + Environment.NewLine;
                                    szDetails += "Supported Version: Protection Info Implementation Version: " + ver.ProtectionInfoVersion.ImplementVersion.Core + " : " + ver.ProtectionInfoVersion.ImplementVersion.DeviceCustomize + Environment.NewLine;
                                    szDetails += "Supported Version: Protection Info Protocol Version: " + ver.ProtectionInfoVersion.ProtocolVersion.Major.ToString() + "." + ver.ProtectionInfoVersion.ProtocolVersion.Minor.ToString() + Environment.NewLine;
                                }
                            }
                            szDetails += "TimeStamp: " + devMFP.Configuration.TimeStamp + Environment.NewLine;
                            szDetails += "OpenAPI Version: " + devMFP.Configuration.Version.Implement + " " + devMFP.Configuration.Version.Major.ToString() + "." + devMFP.Configuration.Version.Minor.ToString() + Environment.NewLine;
                            if (devMFP.device != null)
                            {
                                szDetails += "DeviceType: " + devMFP.device.deviceType + Environment.NewLine;
                                szDetails += "Friendly Name: " + devMFP.device.friendlyName + Environment.NewLine;
                                szDetails += "Manufacturer: " + devMFP.device.manufacturer + Environment.NewLine;
                                szDetails += "Model Name: " + devMFP.device.modelName + Environment.NewLine;
                                foreach (KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.service srv in devMFP.device.serviceList.service)
                                {
                                    szDetails += "Service ID: " + srv.serviceId + Environment.NewLine;
                                    szDetails += "    Type: " + srv.serviceType + Environment.NewLine;
                                    szDetails += "    ControlURL: " + srv.controlURL + Environment.NewLine;
                                    szDetails += "    EventSubURL: " + srv.eventSubURL + Environment.NewLine;
                                    szDetails += "    SCPDURL: " + srv.SCPDURL + Environment.NewLine;
                                }
                                szDetails += "UDN: " + devMFP.device.UDN + Environment.NewLine;
                            }
                            else
                            {
                                szDetails += "No device info";
                            }
                            if (devMFP.specVersion != null)
                                szDetails += "Spec Version: " + devMFP.specVersion.major.ToString() + "." + devMFP.specVersion.minor.ToString() + Environment.NewLine;
                            else
                                szDetails += "Spec Version: <None>" + Environment.NewLine;
                            if (devMFP.External != null)
                            {
                                if (int.Parse(devMFP.External.AuthenticationDeviceList.ArraySize) > 0)
                                {
                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.AuthenticationDevice dev in devMFP.External.AuthenticationDeviceList.AuthenticationDevice)
                                    {
                                        szDetails += "Authentication Device: " + dev.Product + " (" + dev.Manufacturer + ") FW:" + dev.FwVersion + " SN:" + dev.SerialNum + Environment.NewLine;
                                        textBoxAuthDevices.Text += dev.Product + " (" + dev.Manufacturer + ") FW:" + dev.FwVersion + " SN:" + dev.SerialNum + Environment.NewLine;
                                    }
                                }
                                else
                                {
                                    szDetails += "Authentication Devices: None" + Environment.NewLine;
                                }
                                if (int.Parse(devMFP.External.FunctionInfoList.ArraySize) > 0)
                                {
                                    TreeNode tnRoot = new TreeNode();

                                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.FunctionInfo func in devMFP.External.FunctionInfoList.FunctionInfo)
                                    {
                                        string szVersion = func.FunctionVersion.Major.ToString() + "." + func.FunctionVersion.Minor.ToString();

                                        szDetails += "Supported subAPI: " + func.FunctionName + " " + szVersion + Environment.NewLine;
                                        try
                                        {
                                            if (!tnRoot.Nodes.ContainsKey(func.FunctionName))
                                            {
                                                TreeNode tn = new TreeNode(func.FunctionName);
                                                tn.Name = func.FunctionName;
                                                tn.Nodes.Add(string.Empty);
                                                tnRoot.Nodes.Add(tn);
                                            }
                                            if (!tnRoot.Nodes[func.FunctionName].Nodes[0].Text.Contains(szVersion))
                                            {
                                                string szVers = tnRoot.Nodes[func.FunctionName].Nodes[0].Text;
                                                szVers += "," + szVersion;

                                                //Order the list
                                                List<string> lVers = new List<string>(szVers.Split(new string[] { ",", " " }, StringSplitOptions.RemoveEmptyEntries));
                                                lVers.Sort();
                                                tnRoot.Nodes[func.FunctionName].Nodes[0].Text = string.Empty;
                                                foreach (string szVer in lVers)
                                                {
                                                    if (tnRoot.Nodes[func.FunctionName].Nodes[0].Text == string.Empty)
                                                        tnRoot.Nodes[func.FunctionName].Nodes[0].Text += szVer;
                                                    else
                                                        tnRoot.Nodes[func.FunctionName].Nodes[0].Text += ", " + szVer;
                                                }
                                            }
                                        }
                                        catch (Exception ex)
                                        {
                                            MessageBox.Show(ex.Message);
                                        }
                                    }
                                    if (this.InvokeRequired)
                                        this.Invoke(new TreeNodeSetDelegate(TreeNodeSet), tnRoot);
                                    else
                                        TreeNodeSet(tnRoot);
                                }
                                else
                                {
                                    szDetails += "Supported subAPIs: None" + Environment.NewLine;
                                }
                            }
                            else
                            {
                                szDetails += "External OpenAPI Support: None" + Environment.NewLine;
                                optionStateExternalOpenAPIEnabled.Checked = false;
                            }
                        }
                        else
                        {
                            bGotDeviceDescription = false;
                        }
                    }
                    catch(Exception)
                    {
                        //We were unable to get the DeviceDescription - most likely because the MFP's HTTP interface is disabled or the IP address is incorrect
                        bGotDeviceDescription = false;
                    }
                    #endregion
                    CheckIOptionAndExtendedMemory(devMFP);

                    tabControlMFPInfo.Enabled = (bGotDeviceDescription || bGotDeviceInfo); //We got at least one
                    
                    if (bGotDeviceDescription)
                    {
                        try
                        {
                            string szTempFile = Path.GetTempFileName();
                            File.Delete(szTempFile);
                            szTempFile += ".xml";
                            File.WriteAllText(szTempFile, devMFP.XmlSerialize());
                            webBrowserDevDesc.Navigate(szTempFile);                            
                            try
                            {
                                while (webBrowserDevDesc.IsBusy)
                                    Thread.Sleep(100);
                            }
                            catch
                            {
                                Thread.Sleep(500);
                            }
                            bool bDeleted = false;
                            int iCounterToGiveUp = 0;
                            while (!bDeleted && iCounterToGiveUp < 50)
                            {
                                try
                                {
                                    // File.Delete(szTempFile);
                                    bDeleted = true;
                                }
                                catch
                                {
                                    Thread.Sleep(250);
                                    iCounterToGiveUp++;
                                }
                            }
                        }
                        catch { }
                    }
                    else
                    {
                        try
                        {
                            webBrowserDevDesc.DocumentText = "<html><body><center><h1>Device Description could not be retrieved</h1></center></body></html>";
                        }
                        catch { }
                    }
                    if (bGotDeviceInfo)
                    {
                        try
                        {
                            string szTempFile = Path.GetTempFileName();
                            File.Delete(szTempFile);
                            szTempFile += ".xml";
                            File.WriteAllText(szTempFile, openAPIHandler.SetupMainControlManager.DeviceInfo.XmlSerialize());
                            webBrowserDevInfo.Navigate(szTempFile);                            
                            try
                            {
                                while (webBrowserDevInfo.IsBusy)
                                    Thread.Sleep(100);
                            }
                            catch
                            {
                                Thread.Sleep(500);
                            }
                            bool bDeleted = false;
                            int iCounterToGiveUp = 0;
                            while (!bDeleted && iCounterToGiveUp < 50)
                            {
                                try
                                {
                                    //File.Delete(szTempFile);
                                    bDeleted = true;
                                }
                                catch
                                {
                                    Thread.Sleep(250);
                                    iCounterToGiveUp++;
                                }
                            }
                        }
                        catch { }
                    }
                    else
                    {
                        try
                        {
                            webBrowserDevInfo.DocumentText = "<html><body><center><h1>Device Info could not be retrieved</h1></center></body></html>";
                        }
                        catch { }
                    }
                    if ((!bGotDeviceDescription && !bGotDeviceInfo))
                    {
                       labelError.Text = "Failed to retrieve anything from the MFP";
                    }                    
                }
                catch
                {
                    Busy(false);
                }
                Busy(false);            
           
        }             
              
        private void ClearAll()
        {           
            textBoxAuthDevices.Clear();
            textBoxInterpreters.Clear();
            treeViewSubAPIs.Nodes.Clear();
            webBrowserDevInfo.DocumentText = string.Empty;
            webBrowserDevDesc.DocumentText = string.Empty;
            foreach (Control c in this.panelDeviceInfo.Controls)
            {
                Type type = c.GetType();
                if (type.IsInstanceOfType(new OpenAPI_App_Manager.Components.MFPDetails.OptionState()))
                    ((OpenAPI_App_Manager.Components.MFPDetails.OptionState)c).State = CheckState.Indeterminate;
            }
            foreach (Control c in this.panelOpenAPIInfo.Controls)
            {
                Type type = c.GetType();
                if (type.IsInstanceOfType(new OpenAPI_App_Manager.Components.MFPDetails.OptionState()))
                    ((OpenAPI_App_Manager.Components.MFPDetails.OptionState)c).State = CheckState.Indeterminate;
            }
            
        }
        
        private delegate void TreeNodeSetDelegate(TreeNode tnRoot);
        private void TreeNodeSet(TreeNode tnRoot)
        {
            foreach (TreeNode tn in tnRoot.Nodes)
                treeViewSubAPIs.Nodes.Add(tn);
            treeViewSubAPIs.ExpandAll();
        }

        private void CheckIOptionAndExtendedMemory(KONICAMINOLTA.OpenAPISDK.SequenceCommonLibrary.IF.DeviceDescriptionNS.DeviceDescription devMFP)
        {
            try
            {
                //This method sets the "IOption and Memory status". iOption and
                //Memory status cannot be retrived via Device description but it can 
                //be indirectly identified by registering a browser based application.
                //In this application we are registering a PullPrint browser application to
                //confirm the status of IOption and extended memory status.

                // The error information is checked with the following cases to confirm the 
                // status
                // a) If 'Disable FEK' error is returned while registering it means 
                //    - IOption is disabled
                // b) If 'There is no Memory' error is returned while registering it means 
                //    - Memory is not available
                // c) If 'Registration LimitOver' error is returned or the applciation is registered 
                // properly means both iOption and Memory is available in the MFP.

                // Stores the registered Application number, when the Application 
                // registration is successful.
                int registerNo = 0;
                
                #region PullBrowserApplicationSettings

                #region "Set the setting of Basic Application Information"

                // Stores the Pull Browser Application registration details
                // which will be sent to the MFP.
                KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.PullBrowserApplicationSetting pullApplicationSetting = new
                    KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.PullBrowserApplicationSetting();

                // Sets the Pull Browser Application Function version
                pullApplicationSetting.SetFunctionVersionType(KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.PullBrowserFunctionVersionType.v1_0);

                #endregion

                #region "Sets the setting of Pull Browser server".

                // An unique number will be assigned to the registered application in MFP
                // This is based on the Application registration type.
                pullApplicationSetting.AutoNumbering = true;

                pullApplicationSetting.Name = "AppManager_TestApp";
                pullApplicationSetting.ID = "PullId";

                // Node is a way to identify unique MFPs.  It also stores other data 
                // that the app might want to know about when dealing with a 
                // particular MFP.
                pullApplicationSetting.Node = "MFPNode";

                // Sets the server application IP Address, Port, SSL Settings and the
                // timeout values where the MFP will sends requests.
                pullApplicationSetting.Server.Address = "192.168.0.1";
                pullApplicationSetting.Server.Port = devMFP.Configuration.OpenApiConfiguration.PortNo;

                // SSL communication is used for the Secured communication process 
                // between MFP and Application.
                pullApplicationSetting.Server.UseSSL = devMFP.Configuration.OpenApiConfiguration.Ssl;

                // Sets the TimeOut property of the Pull Browser server application.
                pullApplicationSetting.Server.Timeout = 120;

                // AppNode is a unique node for PullPrint applications, used for determining
                // that a print job belongs to a particular app.  Because we are registering
                // as a PullPrint type of application, if it's already registered (due to a
                // failure to delete in the past) our error message will relate to this.
                pullApplicationSetting.Function.AppNode = "64D5F1";
                #endregion

                #region Sets the setting of Pull Browser function.

                // Defines the application authentication 
                // Possible Authentication types are
                // None     : No authentication credentials are required for the 
                //            Pull Browser Application login.
                // Silent   : Authentication credentials used for the MFP login
                //            will be used for the Pull Browser Application login.
                pullApplicationSetting.Function.AuthenticationType = KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.ApplicationAuthenticationType.None;

                // Defines the Job notification setting.
                pullApplicationSetting.Function.EnableJobNotify = false;

                #endregion

                #region Sets the setting of BrowserInfo for the Pull Browser.

                // It will be used by the MFP to create unique folder for the
                // registered Pull Browser application to upload its contents.
                pullApplicationSetting.Browser.FolderID = "PullPrintFolder";
                // Sets the Timeout for the Pull Browser.
                pullApplicationSetting.Browser.Timeout = 20;

                // It can be enabled to transfer data using the web browser application.
                pullApplicationSetting.Browser.Cookie = KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.CookieType.NotUseCookies;

                // It is of type Uri.
                // It will be used by the MFP to access the Web Application pages.
                pullApplicationSetting.Browser.Url = new Uri("http://PullWebApp/Virtual/MainForm.aspx");

                #endregion

                #region [FIXED FOR 273-IB Automatically addition of pull print browser app in registered applications list.]
                pullApplicationSetting.BrowserHardKey = null;
                #endregion

                #endregion

                // Registers the the PullPrintBrowser application with the MFP.
                KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult result = openAPIHandler.SetPullBrowserApplication(
                           (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS.PullBrowserApplicationSetting)pullApplicationSetting,
                           out registerNo
                           );

                if (result == KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Nack)
                {
                    //Setting the default as disabled
                    optionStateExtendedMemory.Checked = false;
                    optionStateIOptionLicence.Checked = false;

                    KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.StatusInfoNS.StatusErrorInfo[] statusInfo = openAPIHandler.SetupMainControlManager.LastConnectionError.ErrorInfoList.StatusErrorInfo;
                    string szErrorDesc = statusInfo[0].Result.ErrorInfo.Description;
                    //Checking Whether the MFP having the Extended Memory and Ioption
                    if (szErrorDesc.Contains("Disable FEK"))
                    {
                        // Case when IOption is disabled
                        optionStateExtendedMemory.Checked = true;
                        optionStateIOptionLicence.Checked = false;
                    }
                    else if (szErrorDesc.Contains("There is no Memory"))
                    {
                        // Case when Memory is not available
                        optionStateExtendedMemory.Checked = false;
                        optionStateIOptionLicence.Checked = false;
                    }
                    else if (szErrorDesc.Contains("Registration LimitOver"))
                    {
                        // Case when both IOption and Memory is enabled but no more application can be registered
                        optionStateExtendedMemory.Checked = true;
                        optionStateIOptionLicence.Checked = true;
                    }
                    else if (szErrorDesc.Contains("AppNode"))
                    {
                        // Case when it was previously registered and somehow failed to delete
                        optionStateExtendedMemory.Checked = true;
                        optionStateIOptionLicence.Checked = true;
                        registerNo = openAPIHandler.FindAppNoByName(pullApplicationSetting.Name);
                    }
                }

                // Request message to Delete the registered application from the MFP.        
                if (registerNo > 0)
                {
                    // Case when application is registered properly
                    optionStateExtendedMemory.Checked = true;
                    optionStateIOptionLicence.Checked = true;

                    //If there are any issue with the actual DeleteApplication request with the application number.
                    KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult connectionResult = openAPIHandler.DeleteApplication(registerNo);
                    //connectResult == "Nack" means failure.
                    if (connectionResult == KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Nack)
                    {
                        MessageBox.Show("Fatal error occured while checking for IOption status or Memory status", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        return;
                    }
                }
            }
            catch 
            {
                //Common reasons we may hit this catch are network communication errors such as incorrect IP address, port number etc, or simply that the MFP has OpenAPI disabled
                optionStateExtendedMemory.State = CheckState.Indeterminate;
                optionStateIOptionLicence.State = CheckState.Indeterminate;
            }
        }

        private void MFPDetails_Resize(object sender, EventArgs e)
        {
            tabControlRawData.ItemSize = new Size((tabControlRawData.Width / 2) - 10, 18);
        }

        private void buttonRefresh_Click(object sender, EventArgs e)
        {   
            //Refresh the page         
            //Clear control's data in the form
            ClearAll();
            //Disable the tabcontrol
            tabControlMFPInfo.Enabled = false;
            //Retrieve MFP details and enable the tabcontrol.
            GatherDetails();           
           
        }
    }
}